
#include "stdafx.h"
#include "ccanvas.h"
#include <wchar.h>


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

cCanvas::cCanvas(CSize p_size)
{
    m_size = p_size;
	m_curAttribute = 0x07;
}

cCanvas::~cCanvas()
{
}

BOOL cCanvas::Resize(CSize p_newsize)
{
    m_size = p_newsize;
    Fill(7,32);
    return TRUE;
}


void cCanvas::FlipX(cOptions* p_pOptions)
{
    tCanvasLine pOldLine = new sCanvasElement[m_size.cx];
    tCanvasLine pNewLine = new sCanvasElement[m_size.cx];
    LONG lRow;
    LONG lCol;
	CSingleLock sl(&m_mutex, TRUE);

    for (lRow = 0; lRow < m_size.cy; lRow++)
    {       
        GetLineInternal(CPoint(0,lRow), pOldLine);
        for (lCol = 0; lCol < m_size.cx; lCol++)
        {
			sCanvasElement ce = pOldLine[lCol];
			ce.character = p_pOptions->GetXMap(ce.character);
            pNewLine[m_size.cx-lCol-1] = ce;

        }
        SetLineInternal(CPoint(0,lRow), pNewLine);
    }

    delete [] pOldLine;
    delete [] pNewLine;
}

void cCanvas::FlipY(cOptions* p_pOptions)
{
    tCanvasLine pOldLine = new sCanvasElement[m_size.cx];
    tCanvasLine pNewLine = new sCanvasElement[m_size.cx];
    LONG lRow;
	LONG lCol;
	CSingleLock sl(&m_mutex, TRUE);

    for (lRow = 0; lRow < m_size.cy/2; lRow++)
    {       
        GetLineInternal(CPoint(0,lRow), pOldLine);
        for (lCol = 0; lCol < m_size.cx; lCol++)
        {
			sCanvasElement ce = pOldLine[lCol];
			ce.character = p_pOptions->GetYMap(ce.character);
            pOldLine[lCol] = ce;
        }

		GetLineInternal(CPoint(0,m_size.cy-lRow-1), pNewLine);
        for (lCol = 0; lCol < m_size.cx; lCol++)
        {
			sCanvasElement ce = pNewLine[lCol];
			ce.character = p_pOptions->GetYMap(ce.character);
            pNewLine[lCol] = ce;
        }
        SetLineInternal(CPoint(0,lRow), pNewLine);
        SetLineInternal(CPoint(0,m_size.cy-lRow-1), pOldLine);
    }

    delete [] pOldLine;
    delete [] pNewLine;
}

void cCanvas::Rotate(cOptions* p_pOptions)
{
    cMemCanvas mc(CSize(m_size.cy,m_size.cx));

    CPoint pt;
	for (pt.y = 0; pt.y < m_size.cy; pt.y++)
    {       
	    for (pt.x = 0; pt.x < m_size.cx; pt.x++)
		{       
			sCanvasElement ce = Get(pt);
			ce.character = p_pOptions->GetRotMap(ce.character);
			mc.Set(CPoint(m_size.cy-pt.y-1, pt.x), ce);
		}
	}
	Resize(CSize(m_size.cy, m_size.cx));
	Set(CPoint(0,0), &mc, CPoint(0,0), m_size);
}

void cCanvas::FillBack(tElementPart p_back, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());
	CSingleLock sl(&m_mutex, TRUE);

    if (p_back == CANVAS_TRANSPARENT) return;

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
	    for (pt.x = rect.left; pt.x < rect.right; pt.x++)
		{       
			SetBackInternal(pt, (UCHAR)p_back);
		}
	}
}

void cCanvas::FillFore(tElementPart p_fore, CRect p_rectSelect)
{
	CSingleLock sl(&m_mutex, TRUE);
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());
    if (p_fore == CANVAS_TRANSPARENT) return;

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
	    for (pt.x = rect.left; pt.x < rect.right; pt.x++)
		{       
			SetForeInternal(pt, (UCHAR)p_fore);
		}
	}
}

void cCanvas::FillAttr(tElementPart p_attribute, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
		SetLine(pt, p_attribute, CANVAS_TRANSPARENT, rect.Width());
	}
}

void cCanvas::FillChar(tElementPart p_character, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
		SetLine(pt, CANVAS_TRANSPARENT, p_character, rect.Width());
	}
}

void cCanvas::Fill(tElementPart p_attribute, tElementPart p_character, CRect p_rectSelect)
{
	CRect rect = (p_rectSelect.IsRectNull()) ? CRect(CPoint(0,0), m_size) : p_rectSelect;
	CPoint pt(rect.TopLeft());

    for (pt.y = rect.top; pt.y < rect.bottom; pt.y++)
    {       
        SetLine(pt, p_attribute, p_character, rect.Width());
    }
}

void cCanvas::Set(cCanvas* p_pCanvas)
{
    Resize(p_pCanvas->m_size);
    LONG lRow;
    LONG lCol;

    for (lRow = 0; lRow < m_size.cy; lRow++)
    {       
        for (lCol = 0; lCol < m_size.cx; lCol++)
        {
            Set(CPoint(lCol,lRow),p_pCanvas->Get(CPoint(lCol, lRow)));
        }
    }
}
void cCanvas::SetLine(CPoint p_point, const tCanvasLine p_line, LONG p_lWidth)
{
	CSingleLock sl(&m_mutex, TRUE);
	SetLineInternal(p_point, p_line, p_lWidth);
}
void cCanvas::GetLine(CPoint p_point, tCanvasLine p_line, LONG p_lWidth)
{
	CSingleLock sl(&m_mutex, TRUE);
	GetLineInternal(p_point, p_line, p_lWidth);
}

void cCanvas::Set(CPoint p_source, cCanvas* p_pCanvas, CPoint p_dest, CSize p_size)
{
	CSingleLock sl(&m_mutex, TRUE);
	CSingleLock sl2(&p_pCanvas->m_mutex, FALSE);
	if (p_pCanvas != this) sl2.Lock();

    tCanvasLine pLine = new sCanvasElement[p_size.cx];
    LONG lRow;
    if (p_dest.x + p_size.cx > GetSize().cx) p_size.cx = GetSize().cx - p_dest.x;
    if (p_dest.y + p_size.cy > GetSize().cy) p_size.cy = GetSize().cy - p_dest.y;

    if (p_source.x + p_size.cx > p_pCanvas->GetSize().cx) p_size.cx = p_pCanvas->GetSize().cx - p_source.x;
    if (p_source.y + p_size.cy > p_pCanvas->GetSize().cy) p_size.cy = p_pCanvas->GetSize().cy - p_source.y;
	for (lRow = 0; lRow < p_size.cy; lRow++)
	{       
		p_pCanvas->GetLineInternal(CPoint(p_source.x,p_source.y+lRow), pLine, p_size.cx);
		SetLineInternal(CPoint(p_dest.x,p_dest.y+lRow), pLine, p_size.cx);
	}
	if (p_pCanvas != this) sl2.Unlock();

    delete [] pLine;
}

void cCanvas::Set(CPoint p_source, cCanvas* p_pCanvas, CPoint p_dest, CSize p_size, cOptions* p_pOptions, BOOL p_bUnder, BOOL p_bTransparent)
{
	CSingleLock sl(&m_mutex, TRUE);
	CSingleLock sl2(&p_pCanvas->m_mutex, FALSE);
	if (p_pCanvas != this) sl2.Lock();
    tCanvasLine pLine = new sCanvasElement[p_size.cx];
    LONG lRow;
	LONG lCol;
    if (p_dest.x + p_size.cx > GetSize().cx) p_size.cx = GetSize().cx - p_dest.x;
    if (p_dest.y + p_size.cy > GetSize().cy) p_size.cy = GetSize().cy - p_dest.y;

    if (p_source.x + p_size.cx > p_pCanvas->GetSize().cx) p_size.cx = p_pCanvas->GetSize().cx - p_source.x;
    if (p_source.y + p_size.cy > p_pCanvas->GetSize().cy) p_size.cy = p_pCanvas->GetSize().cy - p_source.y;
	if (p_bUnder || p_bTransparent)
	{
	    tCanvasLine pOriginalLine = new sCanvasElement[p_size.cx];
		for (lRow = 0; lRow < p_size.cy; lRow++)
		{       
			p_pCanvas->GetLineInternal(CPoint(p_source.x,p_source.y+lRow), pLine, p_size.cx);
			GetLineInternal(CPoint(p_dest.x,p_dest.y+lRow), pOriginalLine, p_size.cx);
			for (lCol = 0; lCol < p_size.cx; lCol++)
			{       
				sCanvasElement& ce = pOriginalLine[lCol];
				sCanvasElement& ceBuf = pLine[lCol];
				BOOL bPaste = TRUE;
				if (p_bUnder)
					bPaste &= (ce.character == 32 && (ce.GetBack() == 0 || (ce.GetBack() == 8 && p_pOptions->GetBackground8())));
				if (p_bTransparent)
					bPaste &= !(ceBuf.character == 32 && (ceBuf.GetBack() == 0 || (ceBuf.GetBack() == 8 && p_pOptions->GetBackground8())));

                if (bPaste) ce = ceBuf;
			}
			SetLineInternal(CPoint(p_dest.x,p_dest.y+lRow), pOriginalLine, p_size.cx);
		}
		delete [] pOriginalLine;
	}
	else 
	{
		for (lRow = 0; lRow < p_size.cy; lRow++)
		{       
			p_pCanvas->GetLineInternal(CPoint(p_source.x,p_source.y+lRow), pLine, p_size.cx);
			SetLineInternal(CPoint(p_dest.x,p_dest.y+lRow), pLine, p_size.cx);
		}
	}
	if (p_pCanvas != this) sl2.Unlock();

    delete [] pLine;
}

void cCanvas::Set(CPoint p_source, cCanvas* p_pCanvas, CPoint p_dest, LONG p_lWidth)
{
    LONG lCol;
	CSingleLock sl(&m_mutex, TRUE);
	CSingleLock sl2(&p_pCanvas->m_mutex, FALSE);
	if (p_pCanvas != this) sl2.Lock();
    for (lCol = 0; lCol < p_lWidth; lCol++)
    {
		CPoint psource = CPoint(p_dest.x+lCol,p_dest.y);
		CPoint pdest = CPoint(p_source.x+lCol,p_source.y);
	    SetAttrInternal(pdest, p_pCanvas->GetAttrInternal(psource));
		SetCharInternal(pdest, p_pCanvas->GetCharInternal(psource));
    }
	if (p_pCanvas != this) sl2.Unlock();
}

void cCanvas::Set(CPoint p_point, sCanvasElement p_element)
{
	CSingleLock sl(&m_mutex, TRUE);
    SetAttrInternal(p_point, p_element.attribute);
    SetCharInternal(p_point, p_element.character);
}

void cCanvas::Set(CPoint p_point, tElementPart p_attribute, tElementPart p_character)
{
	CSingleLock sl(&m_mutex, TRUE);
    if (p_attribute != CANVAS_TRANSPARENT) SetAttrInternal(p_point, (UCHAR)p_attribute);
    if (p_character != CANVAS_TRANSPARENT) SetCharInternal(p_point, (UCHAR)p_character);
}

void cCanvas::Set(CRect p_rect, sCanvasElement p_element)
{
    CPoint pt;
	CSingleLock sl(&m_mutex, TRUE);
    for (pt.y=p_rect.top; pt.y<p_rect.bottom; pt.y++)
    {
        for (pt.x=p_rect.left; pt.x<p_rect.right; pt.x++)
        {
			if (p_element.attribute != CANVAS_TRANSPARENT) SetAttrInternal(pt, p_element.attribute);
			if (p_element.character != CANVAS_TRANSPARENT) SetCharInternal(pt, p_element.character);
        }
    }
}

void cCanvas::Set(CRect p_rect, tElementPart p_attribute, tElementPart p_character)
{
    CPoint pt;
	CSingleLock sl(&m_mutex, TRUE);
    for (pt.y=p_rect.top; pt.y<p_rect.bottom; pt.y++)
    {
        for (pt.x=p_rect.left; pt.x<p_rect.right; pt.x++)
        {
			if (p_attribute != CANVAS_TRANSPARENT) SetAttrInternal(pt, (UCHAR)p_attribute);
			if (p_character != CANVAS_TRANSPARENT) SetCharInternal(pt, (UCHAR)p_character);
        }
    }
}


sCanvasElement cCanvas::Get(CPoint p_point)
{
    sCanvasElement elem;
	CSingleLock sl(&m_mutex, TRUE);
    elem.attribute = GetAttrInternal(p_point);
    elem.character = GetCharInternal(p_point);
    return elem;
}

UCHAR cCanvas::GetAttr(CPoint p_point)
{
	CSingleLock sl(&m_mutex, TRUE);
    return GetAttrInternal(p_point);
}


void cCanvas::SetAttr(CPoint p_point, tElementPart p_attribute)
{
    if (p_attribute != CANVAS_TRANSPARENT)
    {
		CSingleLock sl(&m_mutex, TRUE);
		SetAttrInternal(p_point, (UCHAR)p_attribute);
    }
}

void cCanvas::SetChar(CPoint p_point, tElementPart p_character)
{
    if (p_character != CANVAS_TRANSPARENT)
    {
		CSingleLock sl(&m_mutex, TRUE);
        SetCharInternal(p_point, (UCHAR)p_character);
        if (p_character >= 256)
        {
            SetAttrInternal(p_point,(UCHAR)(GetAttrInternal(p_point) & 0x08));
        }
    }
}

void cCanvas::SetFore(CPoint p_point, tElementPart p_fore)
{
    if (p_fore != CANVAS_TRANSPARENT)
    {
		CSingleLock sl(&m_mutex, TRUE);
        SetForeInternal(p_point, (UCHAR)p_fore);
    }
}

void cCanvas::SetBack(CPoint p_point, tElementPart p_back)
{
    if (p_back != CANVAS_TRANSPARENT)
    {
		CSingleLock sl(&m_mutex, TRUE);
        SetBackInternal(p_point, (UCHAR)p_back);
    }
}

UCHAR cCanvas::GetFore(CPoint p_point)
{
	CSingleLock sl(&m_mutex, TRUE);
    return GetAttrInternal(p_point) & 0xF;
}

UCHAR cCanvas::GetBack(CPoint p_point)
{
	CSingleLock sl(&m_mutex, TRUE);
    return GetAttrInternal(p_point) >> 4;
}


void cCanvas::SetLine(CPoint p_point, tElementPart p_attribute, tElementPart p_character, LONG p_lWidth)
{
    LONG lCol;
    p_lWidth = (p_lWidth == 0) ? GetSize().cx : p_lWidth;
	CSingleLock sl(&m_mutex, TRUE);
    for (lCol = 0; lCol < p_lWidth; lCol++)
    {
		CPoint pt = p_point+CSize(lCol,0);
		if (p_attribute != CANVAS_TRANSPARENT) SetAttrInternal(pt, (UCHAR)p_attribute);
		if (p_character != CANVAS_TRANSPARENT) SetCharInternal(pt, (UCHAR)p_character);
    }
}

void cCanvas::SetLine(CPoint p_point, sCanvasElement p_element, LONG p_lWidth)
{
	SetLine(p_point, p_element.attribute, p_element.character, p_lWidth);
}

void cCanvas::InsertLine(int p_iLine)
{
	LONG lRow;
	LONG lCol;
	sCanvasElement element;
	for (lRow = GetSize().cy-1; lRow > p_iLine; lRow--)
	{
		for (lCol = 0; lCol < GetSize().cx; lCol++)
		{
			element = Get(CPoint(lCol, lRow-1));
			Set(CPoint(lCol, lRow), element);
		}
	}
	SetLine(CPoint(0,p_iLine), 7, 32);
}
void cCanvas::DeleteLine(int p_iLine)
{
	LONG lRow;
	LONG lCol;
	sCanvasElement element;
	for (lRow = p_iLine; lRow < GetSize().cy-1; lRow++)
	{
		for (lCol = 0; lCol < GetSize().cx; lCol++)
		{
			element = Get(CPoint(lCol, lRow+1));
			Set(CPoint(lCol, lRow), element);
		}
	}
	SetLine(CPoint(0,GetSize().cy-1), 7, 32);
}

void cCanvas::Delete(CRect p_rectSelect)
{
	p_rectSelect.NormalizeRect();
	cMemCanvas mc(CSize(m_size.cx, p_rectSelect.Height()));
	mc.Set(CPoint(p_rectSelect.right, p_rectSelect.top), this, CPoint(0,0), CSize(m_size.cx-p_rectSelect.Width(), p_rectSelect.Height()));
	Set(CPoint(0,0), &mc, p_rectSelect.TopLeft(), CSize(m_size.cx, p_rectSelect.Height()));
}


void cCanvas::SetColumn(CPoint p_point, tElementPart p_attribute, tElementPart p_character, LONG p_lHeight)
{
    LONG lRow;
    p_lHeight = (p_lHeight == 0) ? GetSize().cy : p_lHeight;
    for (lRow = 0; lRow < p_lHeight; lRow++)
    {
        Set(p_point+CSize(0,lRow),p_attribute, p_character);
    }
}
void cCanvas::SetColumn(CPoint p_point, sCanvasElement p_element, LONG p_lHeight)
{
	SetColumn(p_point, p_element.attribute, p_element.character, p_lHeight);
}




int cCanvas::FindEndX(int p_iLine, int p_iStart, int p_iEnd)
{
	sCanvasElement element;
	if (p_iEnd == 0) p_iEnd = GetSize().cx;
    for (int lCol = p_iEnd-1; lCol >= p_iStart; lCol--)
	{
		element = Get(CPoint(lCol, p_iLine));
        if ((element.character != ' ' && element.character != 0) || (element.attribute > 15))
        {
            return lCol;
        }
	}
    return -1;
}

int cCanvas::FindEndY()
{
    for (int lRow = GetSize().cy-1; lRow >= 0; lRow--)
    {
        if (FindEndX(lRow) != -1)
        {
            return lRow;
        }
    }
    return -1;
}


void cCanvas::DeleteChar(CPoint p_point, BOOL p_bInsert)
{
	if (p_bInsert)
	{
	    if (p_point.x < GetSize().cx-1)
		{
			CSize size(GetSize().cx-p_point.x-1,1);
			cMemCanvas mc(size);
			mc.Set(p_point+CSize(1,0),this,CPoint(0,0),size);
			Set(CPoint(0,0),&mc,p_point,size);
		}
		Set(CPoint(GetSize().cx-1,p_point.y), 7, 32);
	}
	else
	{
		Set(p_point, 7, 32);
	}
}

void cCanvas::InsertChar(CPoint p_point, sCanvasElement p_element, BOOL p_bInsert)
{
	if (p_bInsert && p_point.x < GetSize().cx-1)
	{
	    CSize size(GetSize().cx-p_point.x-1,1);
		cMemCanvas mc(size);
		mc.Set(p_point+CSize(1,0),this,CPoint(0,0),size);
		Set(p_point,this,p_point+CSize(1,0),size);
	}
	Set(p_point, p_element);
}


// cMemCanvas

cMemCanvas::cMemCanvas(CSize p_size) :
    cCanvas(p_size)
{
    m_pData = NULL;
    Resize(p_size);
}

cMemCanvas::~cMemCanvas()
{
    CSingleLock sl(&m_mutex, TRUE);
    int iCount;
    if (m_pData)
    {
        for (iCount=0; iCount<m_size.cy; iCount++)
        {
            if (m_pData[iCount]) delete [] m_pData[iCount];
        }
        delete [] m_pData;
        m_pData = NULL;
    }
}

BOOL cMemCanvas::Resize(CSize p_newsize)
{
    int iCount;
    
    CSingleLock sl(&m_mutex, TRUE);
    if (m_pData)
    {
        for (iCount=0; iCount<m_size.cy; iCount++)
        {
            if (m_pData[iCount]) delete [] m_pData[iCount];
        }
        delete [] m_pData;
        m_pData = NULL;
    }
    m_pData = new tCanvasLine[p_newsize.cy];
    for (iCount=0; iCount<p_newsize.cy; iCount++)
    {
        m_pData[iCount] = NULL; //new sCanvasElement[p_newsize.cx];
    }
	cCanvas::Resize(p_newsize);
	return TRUE;
}

void cMemCanvas::SetAttrInternal(CPoint p_point, UCHAR p_ucAttribute)
{
	if (p_point.y < m_size.cy && p_point.x < m_size.cx)
	{
		if (!m_pData[p_point.y] && p_ucAttribute != 7)
		{
			m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
			wmemset((wchar_t*)m_pData[p_point.y], 0x2007, GetSize().cx);
		}
		if (m_pData[p_point.y]) m_pData[p_point.y][p_point.x].attribute = p_ucAttribute;
	}
}

void cMemCanvas::SetCharInternal(CPoint p_point, UCHAR p_ucCharacter)
{
	if (p_point.y < m_size.cy && p_point.x < m_size.cx)
	{
		if (!m_pData[p_point.y] && p_ucCharacter != 32)
		{
			m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
			wmemset((wchar_t*)m_pData[p_point.y], 0x2007, GetSize().cx);
		}
		if (m_pData[p_point.y]) m_pData[p_point.y][p_point.x].character = p_ucCharacter;
	}
}

void cMemCanvas::SetForeInternal(CPoint p_point, UCHAR p_ucFore)
{
    {
        if (!m_pData[p_point.y] && p_ucFore != 7)
        {
            m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
            wmemset((wchar_t*)m_pData[p_point.y], 0x2007, GetSize().cx);
        
        }
        if (m_pData[p_point.y]) 
        {
            sCanvasElement* pElem = &m_pData[p_point.y][p_point.x];
            pElem->attribute = (pElem->attribute & 0xF0) | (p_ucFore & 0x0F);
        }
    }
}

void cMemCanvas::SetBackInternal(CPoint p_point, UCHAR p_ucBack)
{
    if (!m_pData[p_point.y] && p_ucBack != 0)
    {
        m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
        wmemset((wchar_t*)m_pData[p_point.y], 0x2007, GetSize().cx);
    }
    if (m_pData[p_point.y]) 
    {
        sCanvasElement* pElem = &m_pData[p_point.y][p_point.x];
        pElem->attribute = (pElem->attribute & 0x0F) | (p_ucBack << 4);
    }
}

UCHAR cMemCanvas::GetAttrInternal(CPoint p_point)
{
    if (!m_pData[p_point.y])
    {
        return 7;
    }
    return m_pData[p_point.y][p_point.x].attribute;
}

UCHAR cMemCanvas::GetCharInternal(CPoint p_point)
{
    if (!m_pData[p_point.y])
    {
        return 32;
    }
    return m_pData[p_point.y][p_point.x].character;
}

cCanvas* cMemCanvas::Create(CSize p_size)
{
    return new cMemCanvas(p_size);
}

void cMemCanvas::GetLineInternal(CPoint p_point, tCanvasLine p_line, LONG p_lWidth)
{
    p_lWidth = (p_lWidth == 0) ? GetSize().cx : p_lWidth;
	if (m_pData[p_point.y])
	{

		wmemcpy((wchar_t*)&p_line[0], (wchar_t*)&m_pData[p_point.y][p_point.x], p_lWidth);
	}
	else
	{
		wmemset((wchar_t*)&p_line[0], (wchar_t)0x2007, p_lWidth);
	}	
}

void cMemCanvas::SetLineInternal(CPoint p_point, const tCanvasLine p_line, LONG p_lWidth)
{
    p_lWidth = (p_lWidth == 0) ? GetSize().cx : p_lWidth;
	if (!m_pData[p_point.y]) m_pData[p_point.y] = new sCanvasElement[GetSize().cx];
	wmemcpy((wchar_t*)&m_pData[p_point.y][p_point.x], (wchar_t*)&p_line[0], p_lWidth);
}

void cMemCanvas::SetLine(CPoint p_point, tElementPart p_attribute, tElementPart p_character, LONG p_lWidth)
{
    if (p_attribute == 7 && p_character == 32 && ((p_lWidth == 0) || (p_lWidth == m_size.cx && p_point.x == 0)))
    {
        CSingleLock sl(&m_mutex, TRUE);
        //if (sl.IsLocked())
        {
            if (m_pData[p_point.y])
            {
                delete [] m_pData[p_point.y];
                m_pData[p_point.y] = NULL;
            }
        }
    }
    else cCanvas::SetLine(p_point, p_attribute, p_character, p_lWidth);
}

void cMemCanvas::InsertLine(int p_iLine)
{
    CSingleLock sl(&m_mutex, TRUE);
    //if (sl.IsLocked())
    {
	    LONG lRow;
	    if (m_pData[m_size.cy-1]) delete [] m_pData[m_size.cy-1];
	    for (lRow = m_size.cy-1; lRow > p_iLine; lRow--)
	    {
		    m_pData[lRow] = m_pData[lRow-1];
	    }
	    m_pData[p_iLine] = NULL; 
    }
	//SetLine(CPoint(0,p_iLine), 7, 32);
}

void cMemCanvas::DeleteLine(int p_iLine)
{
	if (p_iLine < 0 || p_iLine >= m_size.cy) return;
    CSingleLock sl(&m_mutex, TRUE);
	//if (sl.IsLocked())
	{
	    LONG lRow;
	    if (m_pData[p_iLine]) delete [] m_pData[p_iLine];
	    for (lRow = p_iLine; lRow < m_size.cy-1; lRow++)
	    {
		    m_pData[lRow] = m_pData[lRow+1];
	    }
        m_pData[m_size.cy-1] = NULL;
	}
}

void cMemCanvas::InsertColumn(int p_iColumn)
{
    CSingleLock sl(&m_mutex, TRUE);
	//if (sl.IsLocked())
	{
	    LONG lRow;
		LONG lLastY = FindEndY();
	    for (lRow = 0; lRow < lLastY+1; lRow++)
	    {
		    if (m_pData[lRow])
			{
				sCanvasElement* pCanvasLine = new sCanvasElement[GetSize().cx];
				if (p_iColumn > 0) wmemcpy((wchar_t*)&pCanvasLine[0], (wchar_t*)&m_pData[lRow][0], p_iColumn);
				if (p_iColumn < m_size.cy-1) wmemcpy((wchar_t*)&pCanvasLine[p_iColumn+1], (wchar_t*)&m_pData[lRow][p_iColumn], m_size.cx-p_iColumn-1);
				pCanvasLine[p_iColumn].value = 0x2007;
				delete [] m_pData[lRow];
				m_pData[lRow] = pCanvasLine;
			}
	    }
	}
}

void cMemCanvas::DeleteColumn(int p_iColumn)
{
    CSingleLock sl(&m_mutex, TRUE);
	//if (sl.IsLocked())
	{
	    LONG lRow;
		LONG lLastY = FindEndY();
	    for (lRow = 0; lRow < lLastY+1; lRow++)
	    {
		    if (m_pData[lRow])
			{
				sCanvasElement* pCanvasLine = new sCanvasElement[GetSize().cx];
				if (p_iColumn > 0) wmemcpy((wchar_t*)&pCanvasLine[0], (wchar_t*)&m_pData[lRow][0], p_iColumn);
				if (p_iColumn < m_size.cy-1) wmemcpy((wchar_t*)&pCanvasLine[p_iColumn], (wchar_t*)&m_pData[lRow][p_iColumn+1], m_size.cx-p_iColumn-1);
				pCanvasLine[m_size.cx-1].value = 0x2007;
				delete [] m_pData[lRow];
				m_pData[lRow] = pCanvasLine;
			}
	    }
	}
}

int cMemCanvas::FindEndY()
{
    CSingleLock sl(&m_mutex, TRUE);
	//if (sl.IsLocked())
	{
	    LONG lRow;
		for (lRow = m_size.cy-1; lRow >= 0; lRow--)
		{
			if (m_pData[lRow] && FindEndX(lRow) != -1) return lRow;
		}
	}
	return -1;
}

